/******************************************************************************
* File:    cpu.h
*
* Purpose: Provide platform-specific routines for the MC56F8086 device.
* Author:  Quinten Brooks, Freescale Semiconductor
* Email:   quinten.brooks@freescale.com
* Date:    2007.03.21
*******************************************************************************
* Note: Items exported from ".c" to ".asm" must be prefixed with a "F".
******************************************************************************/

#ifndef __CPU_H
#define __CPU_H     1


///////////////////////////////////////////////////////////////////////////////
// Type Definitions
///////////////////////////////////////////////////////////////////////////////
typedef signed char             int8_t;
typedef signed short int        int16_t;
typedef signed long int         int32_t;

typedef unsigned char           uint8_t;
typedef unsigned short int      uint16_t;
typedef unsigned long int       uint32_t;

typedef volatile int8_t         vint8_t;
typedef volatile int16_t        vint16_t;
typedef volatile int32_t        vint32_t;

typedef volatile uint8_t        vuint8_t;
typedef volatile uint16_t       vuint16_t;
typedef volatile uint32_t       vuint32_t;


///////////////////////////////////////////////////////////////////////////////
// Generic Bit Definitions
///////////////////////////////////////////////////////////////////////////////
#define BIT0    0x1
#define BIT1    0x2
#define BIT2    0x4
#define BIT3    0x8
#define BIT4    0x10
#define BIT5    0x20
#define BIT6    0x40
#define BIT7    0x80
#define BIT8    0x100
#define BIT9    0x200
#define BIT10   0x400
#define BIT11   0x800
#define BIT12   0x1000
#define BIT13   0x2000
#define BIT14   0x4000
#define BIT15   0x8000
#define BIT16   0x10000
#define BIT17   0x20000
#define BIT18   0x40000
#define BIT19   0x80000
#define BIT20   0x100000
#define BIT21   0x200000
#define BIT22   0x400000
#define BIT23   0x800000
#define BIT24   0x1000000
#define BIT25   0x2000000
#define BIT26   0x4000000
#define BIT27   0x8000000
#define BIT28   0x10000000
#define BIT29   0x20000000
#define BIT30   0x40000000
#define BIT31   0x80000000


///////////////////////////////////////////////////////////////////////////////
// Platform-specific defines
///////////////////////////////////////////////////////////////////////////////
#define INTERRUPT                       // Function label for ISRs
#define INTERRUPT_ON    interrupt on    // Use #pragma to return w/ "RTI"
#define INTERRUPT_OFF   interrupt off   // Use #pragam to return w/ "RTS"


///////////////////////////////////////////////////////////////////////////////
// Operating Mode Register (OMR) Bitmasks
///////////////////////////////////////////////////////////////////////////////
#define OMR_NL          BIT15           // Nested Loop
#define OMR_CM          BIT8            // Condition Code Mode
#define OMR_XP          BIT7            // X or P Execute
#define OMR_SD          BIT6            // Stop Delay
#define OMR_R           BIT5            // Rounding
#define OMR_SA          BIT4            // Saturation Mode
#define OMR_EX          BIT3            // External Data Memory
#define OMR_MB          BIT1            // Mode B (Flash Security)
#define OMR_MA          BIT0            // Mode A (Extboot)


///////////////////////////////////////////////////////////////////////////////
// Status Register (SR) Bitmasks
///////////////////////////////////////////////////////////////////////////////
#define SR_LF   BIT15           // Loop Flag
#define SR_P    0x7C00          // Program Counter Extension
#define SR_P4   BIT14           // Program Counter Extension[4]
#define SR_P3   BIT13           // Program Counter Extension[3]
#define SR_P2   BIT12           // Program Counter Extension[2]
#define SR_P1   BIT11           // Program Counter Extension[1]
#define SR_P0   BIT10           // Program Counter Extension[0]
#define SR_I    (BIT9+BIT8)     // Interrupt Mask
#define SR_I1   BIT9            // Interrupt Mask[1]
#define SR_I0   BIT8            // Interrupt Mask[0]
#define SR_SZ   BIT7            // Size
#define SR_L    BIT6            // Limit
#define SR_E    BIT5            // Extension In Use
#define SR_U    BIT4            // Un-normalized
#define SR_N    BIT3            // Negative
#define SR_Z    BIT2            // Zero
#define SR_V    BIT1            // Overflow
#define SR_C    BIT0            // Carry


//////////////////////////////////////////////////////////////////////////////
// Interrupt Priority Level (IPL) Bitmasks
//////////////////////////////////////////////////////////////////////////////
#define IPL0    (0)             // Enable IPL0, IPL1, IPL2, IPL3
#define IPL1    (SR_I0)         // Enable IPL1, IPL2, IPL3
#define IPL2    (SR_I1)         // Enable IPL2, IP3
#define IPL3    (SR_I1+SR_I0)   // Enable IPL3


///////////////////////////////////////////////////////////////////////////////
// Special CPU Routines
///////////////////////////////////////////////////////////////////////////////
#define omr_bfchg(x)            { asm(bfchg #(x),omr); asm(nop); }
#define omr_bfclr(x)            { asm(bfclr #(x),omr); asm(nop); }
#define omr_bfset(x)            { asm(bfset #(x),omr); asm(nop); }
#define omr_write(x)            { asm(move.w #(x),omr); asm(nop); }


///////////////////////////////////////////////////////////////////////////////
// Miscellaneous defines
///////////////////////////////////////////////////////////////////////////////

#define bitfield(x,y)		(##x##_0 * y)                   // Ex: 
#define bitset(x,y)             ((x) |= (y))                    // Ex: bitset(GPIOA_DR, BIT0)
#define bitclr(x,y)             ((x) &= ~(y))                   // Ex: bitclr(SCI_CR, BIT0);
#define bittest(x,y)            ((((x) & (y)) == (y)) ? 1 : 0)	// Ex: bittest(SCI_STAT, SCI_STAT_TDRE)

#ifdef  FALSE
#undef  FALSE
#endif
#define FALSE   (0)

#ifdef  TRUE
#undef  TRUE
#endif
#define TRUE    (1)

#ifdef  NULL
#undef  NULL
#endif
#define NULL    (0)

#ifdef  ON
#undef  ON
#endif
#define ON      (1)

#ifdef  OFF
#undef  OFF
#endif
#define OFF     (0)


///////////////////////////////////////////////////////////////////////////////
// Address markers from linker command file
//
// Note: The following variables are prefixed with "_" in the linker
//       command file.  The leading "_" is not used in ".c" files.
///////////////////////////////////////////////////////////////////////////////
extern uint32_t bss_end;              // Zero-initialized data (end)
extern uint32_t bss_start;            // Zero-initialized data (start)
extern uint32_t bss_size;             // Zero-initialized data (size)
extern uint32_t mirror_end;           // RAM-Mirrored into ROM (end)
extern uint32_t mirror_start;         // RAM-Mirrored into ROM (start)
extern uint32_t mirror_size;          // RAM-Mirrored into ROM (size)
extern uint32_t pflash_start;         // Program Flash Low Boundary
extern uint32_t pflash_end;           // Program Flash High Boundary
extern uint32_t pflash_size;          // Program Flash used (in words)
extern uint32_t pflash_free;          // Program Flash RAM Mirror
extern uint32_t pram_start;           // Program RAM Low Boundary
extern uint32_t pram_end;             // Program RAM High Boundary
extern uint32_t pram_size;            // Program RAM used (in words)
extern uint32_t xram_start;           // Data RAM Low Boundary
extern uint32_t xram_end;             // Data RAM High Boundary
extern uint32_t xram_size;            // Data RAM used (in words)
extern uint32_t sp_start;             // Data Constants used (in words)
extern uint16_t vba;                  // Data Constants used (in words)


///////////////////////////////////////////////////////////////////////////////
// Use "pragma" to define code and data segments
///////////////////////////////////////////////////////////////////////////////
#pragma define_section CODE_FLASH "CODE_FLASH.text" RX
#pragma define_section CODE_RAM   "CODE_RAM.text"   RWX


///////////////////////////////////////////////////////////////////////////////
// Export functions from "cpu.c"
///////////////////////////////////////////////////////////////////////////////
extern void entry_point(void);
extern void exit_test(void);
extern void init_cpu(void);
extern void init_bss(register *xaddr, register num);
extern void init_periphs(void);
extern void init_ram(register *paddr, register *xaddr, register num);
extern void set_clk_external(void);
extern void enable_pll(void);


///////////////////////////////////////////////////////////////////////////////
// Structure Definitions 
///////////////////////////////////////////////////////////////////////////////

#ifndef COMPILER_HAS_BOOL
typedef int bool;
#endif

typedef volatile unsigned BITFIELD;

typedef struct {
   unsigned BT0 : 1;
   unsigned BT1 : 1;
   unsigned BT2 : 1;
   unsigned BT3 : 1;
   unsigned BT4 : 1;
   unsigned BT5 : 1;
   unsigned BT6 : 1;
   unsigned BT7 : 1;
   unsigned BT8 : 1;
   unsigned BT9 : 1;
   unsigned BT10 : 1;
   unsigned BT11 : 1;
   unsigned BT12 : 1;
   unsigned BT13 : 1;
   unsigned BT14 : 1;
   unsigned BT15 : 1;
} _regBITS;

typedef union {
  _regBITS bits; 
  uint16_t all;
} generic_16bit;


///////////////////////////////////////////////////////////////////////////////
// Inline Functions
///////////////////////////////////////////////////////////////////////////////

/*****************************************************************************
*
*****************************************************************************/
inline void bfclr_omr(register uint16_t mask)
{
   register uint16_t tmp;
   asm {
	not.w	mask            // Get OMR bits to clear
	move.w	omr,tmp         // Get current OMR value
	and.w	mask,tmp        // Get desired OMR value (w/ bits cleared)
	moveu.w	tmp,omr         // Set OMR
	nop
	nop
   }
}

/*****************************************************************************
*
*****************************************************************************/
inline void bfset_omr(register uint16_t mask)
{
   register uint16_t tmp;
   asm {
	move.w	omr,tmp         // Get current OMR value
	or.w	mask,tmp        // Get desired OMR value (w/ bits set)
	moveu.w	tmp,omr         // Set OMR
	nop
	nop
   }
}

/*****************************************************************************
*
*****************************************************************************/
inline uint16_t pmem_read(register uint16_t *addr)
{
   register uint16_t data;

   asm(move.w p:(addr)+,data);
   return data;
}

/*****************************************************************************
*
*****************************************************************************/
inline void pmem_write(register uint16_t* addr, register uint16_t data)
{
   asm(move.w data,p:(addr)+);
}

/*****************************************************************************
*
*****************************************************************************/
inline uint16_t read_omr(void)
{
   register uint16_t data;
   asm(move.w omr,data);
   return data;
}

/*****************************************************************************
*
*****************************************************************************/
inline uint16_t read_sr(void)
{
   register uint16_t value;

   asm(move.w sr,value);
   return(value);
}

/*****************************************************************************
* sleep ()
*
* Execute NOPs to delay code execution.
* 
* In:  value - Number of NOPs to perform
*
* Out: n/a
*
*****************************************************************************/
inline void sleep(register uint16_t value)
{
   asm(rep value);
   asm(nop);
}

/*****************************************************************************
*
*****************************************************************************/
inline void write_omr(register uint16_t value)
{
   asm(moveu.w value,omr);
   asm(nop);
   asm(nop);
}

/*****************************************************************************
*
*****************************************************************************/
inline void write_sr(register uint16_t value)
{
   asm(moveu.w value,sr);
}

/*****************************************************************************
*
*****************************************************************************/
inline uint16_t xmem_read(register uint16_t *addr)
{
   register uint16_t data;

   asm(move.w x:(addr),data);
   return data;
}

/*****************************************************************************
*
*****************************************************************************/
inline void xmem_write(register uint16_t *addr, register uint16_t data)
{
   asm(move.w  data,x:(addr));
}

#endif  __CPU_H

/* End of "cpu.h" */


